<?php

/**
 * The public-facing functionality of the plugin.
 *
 * @link       https://www.webexpert.gr/
 * @since      1.0.0
 *
 * @package    Webexpert_Woocommerce_Skroutz_Smart_Cart
 * @subpackage Webexpert_Woocommerce_Skroutz_Smart_Cart/public
 */

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the public-facing stylesheet and JavaScript.
 *
 * @package    Webexpert_Woocommerce_Skroutz_Smart_Cart
 * @subpackage Webexpert_Woocommerce_Skroutz_Smart_Cart/public
 * @author     Web Expert <info@webexpert.gr>
 */
class Webexpert_Woocommerce_Skroutz_Smart_Cart_Public {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of the plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {

		$this->plugin_name = $plugin_name;
		$this->version = $version;

	}

	/**
	 * Register the stylesheets for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Webexpert_Woocommerce_Skroutz_Smart_Cart_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Webexpert_Woocommerce_Skroutz_Smart_Cart_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/webexpert-woocommerce-skroutz-smart-cart-public.css', array(), $this->version, 'all' );

	}

	/**
	 * Register the JavaScript for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Webexpert_Woocommerce_Skroutz_Smart_Cart_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Webexpert_Woocommerce_Skroutz_Smart_Cart_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/webexpert-woocommerce-skroutz-smart-cart-public.js', array( 'jquery' ), $this->version, false );

	}

	public function fetch($path,$method='GET',$query=[]) {
		if (empty(get_option('webexpert_skroutz_smart_cart_api_token',null))) {
			return false;
		}

		$ch = curl_init();
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch,CURLOPT_RETURNTRANSFER,1);
		curl_setopt($ch,CURLOPT_HTTPHEADER, array(
			"Authorization: Bearer ".get_option('webexpert_skroutz_smart_cart_api_token',null),
			"Content-Type: application/json",
			"Accept: application/vnd.skroutz+json; version=3.0"
		));

		if ($method=="POST") {
			curl_setopt($ch, CURLOPT_POST, 1);
			curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($query,JSON_UNESCAPED_UNICODE));
			$url="https://api.skroutz.gr/merchants/ecommerce/{$path}";
		}elseif ($method=="PUT") {
			curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "PUT");
			curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
			curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($query,JSON_UNESCAPED_UNICODE));
			$url="https://api.skroutz.gr/merchants/ecommerce/{$path}";
		}else {
			$url="https://api.skroutz.gr/merchants/ecommerce/{$path}";
			if (sizeof($query)>0) {
				$url="https://api.skroutz.gr/merchants/ecommerce/{$path}?".http_build_query($query);
			}
		}

		curl_setopt($ch, CURLOPT_URL,$url);

		return json_decode(curl_exec($ch));
	}

    public function register_smart_cart_fn()
    {
        register_rest_route( 'wc/v3', 'smart_cart', array(
            'methods' => 'POST',
            'callback' => array($this,'process_order'),
            'permission_callback' => '__return_true'
        ));
	}


    function add_my_gateway_class( $methods ) {
        $methods[] = 'WC_Smart_Cart_Gateway';
        return $methods;
    }

	function create_new_order($sc_order,$update=false) {
	    global $woocommerce;
        try {
            $preferred_payment_gateway=get_option('webexpert_skroutz_smart_cart_prefered_payment_gateway','skroutz_smart_cart');
            $preferred_shipping_method=get_option('webexpert_skroutz_smart_cart_prefered_shipping_method','');
            $order_id=wc_get_order_id_by_order_key("SC-{$sc_order->code}");
	        if ($order_id) {
		        $order=wc_get_order($order_id);
		        if ($update==false) {
			        return null;
		        }
	        }else {
		        $order=new WC_Order();
	        }

            $order->set_order_key($sc_order->code);

	        if (in_array($sc_order->state,["cancelled","rejected","expired"])) {
		        $order->set_status("cancelled");
	        }elseif (in_array($sc_order->state,["open"])) {
		        $order->set_status("on-hold");
	        }elseif (in_array($sc_order->state,["dispatched","delivered"])) {
		        $order->set_status("completed");
	        }elseif (in_array($sc_order->state,["returned","partially_returned"])) {
		        $order->set_status("refunded");
	        }elseif (in_array($sc_order->state,["for_return"])) {
		        $order->set_status("for-return");
	        }else {
		        $order->set_status(apply_filters("webexpert_skroutz_smart_cart_custom_processing_status","processing",$order));
	        }

            $email=get_option('webexpert_skroutz_smart_cart_billing_email','');
			if (empty($email)) {
				$email="{$sc_order->customer->id}@skroutz.gr";
			}
            if (get_option('webexpert_skroutz_smart_cart_user_id',-1)>0) {
                $order->set_customer_id(get_option('webexpert_skroutz_smart_cart_user_id',-1));
                $user_data=get_userdata(get_option('webexpert_skroutz_smart_cart_user_id',-1));
                $email=empty(get_option('webexpert_skroutz_smart_cart_billing_email')) ? $user_data->user_email : get_option('webexpert_skroutz_smart_cart_billing_email');
            }
            $order->set_date_created(strtotime($sc_order->created_at));
            $order->set_date_paid(strtotime($sc_order->created_at));
            $order->set_date_modified(strtotime($sc_order->created_at));
            $order->set_created_via( 'skroutz_smart_cart' );
            $order->set_currency( get_woocommerce_currency() );
            $order->set_prices_include_tax( 'yes' === get_option( 'woocommerce_prices_include_tax' ) );
            $order->set_customer_note( $sc_order->comments);

            foreach ( WC()->payment_gateways->payment_gateways() as $method ) {
            	if ($method && isset($method->id) && $method->id==$preferred_payment_gateway) {
		            $order->set_payment_method_title( $method->get_method_title() );
	            }
            }
	        $order->set_payment_method( $preferred_payment_gateway );
	        $order->update_meta_data(apply_filters('webexpert_skroutz_smart_cart_customer_id','skroutz_customer_id'),$sc_order->customer->id);

            $order->set_billing_first_name($sc_order->customer->first_name);
            $order->set_billing_last_name($sc_order->customer->last_name);
            $order->set_billing_address_1($sc_order->customer->address->street_name. ' '. $sc_order->customer->address->street_number);
            $order->set_billing_city($sc_order->customer->address->city);
	        $order->set_billing_postcode($sc_order->customer->address->zip);
            $order->update_meta_data('_skroutz_customer_id',$sc_order->customer->id);
            $order->update_meta_data('_skroutz_pickup_from_collection_point',$sc_order->customer->address->pickup_from_collection_point);
            $order->update_meta_data('_skroutz_collection_point_address',$sc_order->customer->address->collection_point_address);
            $order->update_meta_data('_skroutz_courier',$sc_order->courier);
            $order->update_meta_data('_skroutz_express',$sc_order->express);
            $order->update_meta_data('_skroutz_fulfilled',$sc_order->fulfilled_by_skroutz);
            $order->update_meta_data('_skroutz_dispatch_until',$sc_order->dispatch_until);
            $order->update_meta_data('_skroutz_courier_voucher',$sc_order->courier_voucher);
            $order->update_meta_data('_skroutz_courier_tracking_codes',$sc_order->courier_tracking_codes);
            $order->set_billing_email(str_replace("{customer}",$sc_order->customer->id,$email));
            $order->set_billing_country("GR");

            $order->set_shipping_first_name($sc_order->customer->first_name);
            $order->set_shipping_last_name($sc_order->customer->last_name);
            $order->set_shipping_address_1($sc_order->customer->address->street_name. ' '. $sc_order->customer->address->street_number);
            $order->set_shipping_city($sc_order->customer->address->city);
            $order->set_shipping_postcode($sc_order->customer->address->zip);
            $order->set_shipping_country("GR");

	        $skip=false;
	        if ($update && in_array($sc_order->state,["cancelled","rejected","expired","refunded","delivered","dispatched","completed"])) {
		        $skip=true;
	        }

	        if ($skip===false) {
		        $items = $order->get_items();
		        foreach ( $items as $item ) {
			        $order->remove_item( $item->get_id() );
		        }
		        $items = $order->get_items('shipping');
		        foreach ( $items as $item ) {
			        $order->remove_item( $item->get_id() );
		        }

	            foreach( $sc_order->line_items as $line_item ) {
		            $product=null;
		            $special_id=get_option('we_skroutz_xml_color_id_parent_id',get_option('webexpert_skroutz_smart_cart_special_id',false));
		            $color_parent_attributes=[];
		            if ($special_id) {
			            $try_to_get_parent = explode("-", $line_item->shop_uid);
			            if (sizeof($try_to_get_parent)>1) {
				            $n = sizeof($try_to_get_parent);
				            for ($i = 0; $i < $n; $i++) {
					            $try_to_get_parent[$i] = implode("-", isset($try_to_get_parent[$i-1]) ? [$try_to_get_parent[$i-1], $try_to_get_parent[$i]] : [$try_to_get_parent[$i]]);
					            $product_parent = $try_to_get_parent[$i];
					            if (get_option('webexpert_skroutz_smart_cart_unique_id','sku')=='sku') {
						            $product_id = wc_get_product_id_by_sku($product_parent);
						            if ($product_id) {
							            $product=wc_get_product($product_id);
						            }
					            }else {
						            $product=wc_get_product($product_parent);
					            }
					            if ($product) {
						            for ($z=1;$z<sizeof($try_to_get_parent)-$i;$z++) {
							            $term=get_term($try_to_get_parent[$z+$i]);
							            if (!is_wp_error($term)) {
								            $color_parent_attributes["attribute_{$term->taxonomy}"]=$term->slug;
							            }
						            }
						            break;
					            }
				            }
			            }else{
				            if (get_option('webexpert_skroutz_smart_cart_unique_id','sku')=='sku') {
					            $product_id = wc_get_product_id_by_sku($line_item->shop_uid);
					            if ($product_id) {
						            $product=wc_get_product($product_id);
					            }
				            }else {
					            $product=wc_get_product($line_item->shop_uid);
				            }
			            }
		            }

		            if (!$product) {
			            $product = wc_get_product(apply_filters('webexpert_smart_cart_custom_product',null,$line_item->shop_uid));
		            }

		            if (!$product) {
			            if (get_option('webexpert_skroutz_smart_cart_unique_id','sku')=='sku') {
				            $product_id = wc_get_product_id_by_sku($line_item->shop_uid);
				            if ($product_id) {
					            $product=wc_get_product($product_id);
				            }
			            }else {
				            if (apply_filters('webexpert_smart_cart_custom_uid',null)) {
					            $meta_key=apply_filters('webexpert_smart_cart_custom_uid',null);
					            $args = array(
						            'post_type' => 'product',
						            'post_status' => 'publish',
						            'meta_key' => $meta_key,
						            'meta_value' => $line_item->shop_uid,
						            'meta_compare' => 'IN'
					            );
					            $products = wc_get_products($args);
					            if (is_array($products)) {
						            $product=reset($products);
					            }
				            }else {
					            $product=wc_get_product($line_item->shop_uid);
				            }
			            }
		            }

		            if ($product) {
			            if ($product->is_type('variable')) {
				            if (isset($line_item->size) && (!empty($line_item->size->shop_value) || !empty($line_item->size->value))) {
					            $shop_value=!empty($line_item->size->shop_value) ? $line_item->size->shop_value : $line_item->size->value;
					            $variation_attributes=get_option('webexpert_skroutz_smart_cart_variation_attributes',[]);
					            foreach ($variation_attributes as $variation_attribute) {
						            // trying to obtain term
						            $t=get_term_by('name',$shop_value,wc_attribute_taxonomy_name($variation_attribute));
						            if (!is_wp_error($t)) {
							            $variation_id=$this->webex_find_variation($product->get_id(),array_merge($color_parent_attributes,['attribute_pa_'.$variation_attribute=>$t->slug]));
							            if ($variation_id) {
								            $product=wc_get_product($variation_id);
								            break;
							            }
						            }
					            }
				            }
			            }

			            $args = array(
				            'name'         => $line_item->product_name,
				            'subtotal'     => wc_get_price_excluding_tax($product,['price'=>$line_item->unit_price,'qty'=>$line_item->quantity]),
				            'total'        => wc_get_price_excluding_tax($product,['price'=>$line_item->unit_price,'qty'=>$line_item->quantity]),
				            'quantity'     => $line_item->quantity,
			            );

			            if ('no' === get_option( 'woocommerce_prices_include_tax' )) {
				            $tax_rates = WC_Tax::get_rates( $product->get_tax_class() );
				            if (!empty($tax_rates)) {
					            $tax_rate = reset($tax_rates);
					            $args['subtotal']=$args['subtotal']/(1+$tax_rate['rate']/100);
					            $args['total']=$args['total']/(1+$tax_rate['rate']/100);
				            }
			            }

			            $order_item=$order->add_product( $product,$line_item->quantity, $args );
			            if (!empty($line_item->extra_info)) {
				            wc_update_order_item_meta($order_item,apply_filters('webexpert_smart_cart_custom_extra_info_field','extra_info'), $line_item->extra_info);
			            }
	                }
	            }
	        }

            $order->set_date_created(strtotime($sc_order->created_at));
            $order->set_date_modified(strtotime($sc_order->created_at));
            $order->set_order_key("SC-{$sc_order->code}");
            $order->save();
            update_post_meta($order->get_id(),'_order_key',"SC-{$sc_order->code}");

	        $shipping_items    = (array) $order->get_items('shipping');
	        if ( sizeof( $shipping_items ) == 0 ) {
		        $separate=explode(":",$preferred_shipping_method);
		        $item = new WC_Order_Item_Shipping();

		        $methods=[];
		        $zone = new \WC_Shipping_Zone( 0 );
		        foreach ( $zone->get_shipping_methods() as $shipping_method ) {
			        $id=$shipping_method->id;
			        $id.=!empty($shipping_method->get_instance_id()) ? ':'.$shipping_method->get_instance_id() : '';
			        $methods[$id]=$shipping_method->get_method_title();
		        }

		        $zones = WC_Shipping_Zones::get_zones();
		        foreach ( $zones as $zone ) {
			        $zone = new \WC_Shipping_Zone( $zone['id'] );
			        foreach ( $zone->get_shipping_methods() as $shipping_method ) {
				        $id=$shipping_method->id;
				        $id.=!empty($shipping_method->get_instance_id()) ? ':'.$shipping_method->get_instance_id() : '';
				        $methods[$id]=$shipping_method->get_title();
			        }
		        }

		        if (is_array($separate) && sizeof($separate)==2) {
			        $item->set_method_id( $separate[0] );
			        $item->set_instance_id( $separate[1] );
		        }else {
			        $item->set_method_id( $preferred_shipping_method );
		        }
		        $item->set_method_title($methods[$preferred_shipping_method]);
		        $order->add_item( $item );
		        $order->calculate_totals();
		        $order->save();
	        }

	        if (isset($sc_order->invoice) && $sc_order->invoice=='true') {
		        $order->update_meta_data( apply_filters('webexpert_smart_cart_invoice_toggle','_billing_invoice'), 'y');
		        $order->update_meta_data( apply_filters('webexpert_smart_cart_billing_activity','_billing_activity'), $sc_order->invoice_details->profession);
		        $order->update_meta_data( apply_filters('webexpert_smart_cart_billing_vat_id','_billing_vat_id'), $sc_order->invoice_details->vat_number);
		        $order->update_meta_data(apply_filters('webexpert_smart_cart_billing_tax_office','_billing_tax_office'), $sc_order->invoice_details->doy);

		        $order->set_billing_address_1("{$sc_order->invoice_details->address->street_name} {$sc_order->invoice_details->address->street_number}");
		        $order->set_billing_postcode($sc_order->invoice_details->address->zip);
		        $order->set_billing_city($sc_order->invoice_details->address->city);
		        $order->set_billing_company($sc_order->invoice_details->company);
		        $order->save();
	        }else {
		        $order->update_meta_data( apply_filters('webexpert_smart_cart_invoice_toggle','_billing_invoice'), 'n');
	        }

	        $order_id=$order->save();

            if ($order_id) {
//	            echo "Order #$order_id was imported";
	            do_action( 'webexpert_skroutz_smart_cart_order_inserted', $order_id );
            	return $order_id;
            }
            return false;

        }catch (Exception $e) {
            echo $e->getMessage();
        }
    }

    function webexpert_skroutz_smart_cart_register_for_return_status($order_statuses) {
	    $order_statuses['wc-for-return'] = array(
		    'label'                     => _x( 'For return', 'Order status', 'webexpert-woocommerce-skroutz-smart-cart' ),
		    'public'                    => false,
		    'exclude_from_search'       => false,
		    'show_in_admin_all_list'    => true,
		    'show_in_admin_status_list' => true,
		    'label_count'               => _n_noop( 'For return <span class="count">(%s)</span>', 'For return <span class="count">(%s)</span>', 'webexpert-woocommerce-skroutz-smart-cart' ),
	    );
	    return $order_statuses;
    }

    function webexpert_skroutz_smart_cart_show_for_return_status($order_statuses) {
	    $order_statuses['wc-for-return'] = _x( 'For return', 'Order status', 'webexpert-woocommerce-skroutz-smart-cart' );
	    return $order_statuses;
    }

    function webexpert_skroutz_smart_cart_get_for_return_status_bulk($bulk_actions) {
	    $bulk_actions['mark_for-return'] = 'Change status to custom status';
	    return $bulk_actions;
    }

	function webexpert_webexpert_skroutz_smart_cart_meta_box($post_type, $post) {
		if (strpos(get_post_meta($post->ID,'_order_key',true), 'SC-') !== false) {
			add_meta_box(
				'webexpert_skroutz_smart_cart',
				__('Skroutz Smart Cart', 'webexpert-woocommerce-order-tracking'),
				[$this, 'webexpert_skroutz_smart_cart_metabox_content'],
				'shop_order', 'side', 'core'
			);
		}
	}

	function webexpert_skroutz_smart_cart_metabox_content($post) {
		$order=wc_get_order($post->ID);
		if ($order->get_created_via()=='skroutz_smart_cart') {
			echo "<div class='wrapper'>";
			$order_key=str_replace("SC-","",$order->get_order_key());
			$response=$this->fetch('orders/'.$order_key);
			if ($response->order->state=='open') {
				echo '<div class="webexpert-skroutz-status open"><span>'.__('New','webexpert-woocommerce-skroutz-smart-cart').'</span></div>';
			}elseif ($response->order->state=='accepted') {
				echo '<div class="webexpert-skroutz-status accepted"><span>'.__('To be sent','webexpert-woocommerce-skroutz-smart-cart').'</span></div>';
			}elseif ($response->order->state=='rejected') {
				echo '<div class="webexpert-skroutz-status rejected"><span>'.__('Rejected','webexpert-woocommerce-skroutz-smart-cart').'</span></div>';
			}elseif ($response->order->state=='dispatched') {
				echo '<div class="webexpert-skroutz-status dispatched"><span>'.__('Dispatched','webexpert-woocommerce-skroutz-smart-cart').'</span></div>';
			}elseif ($response->order->state=='delivered') {
				echo '<div class="webexpert-skroutz-status delivered"><span>'.__('Delivered','webexpert-woocommerce-skroutz-smart-cart').'</span></div>';
			}elseif ($response->order->state=='expired') {
				echo '<div class="webexpert-skroutz-status expired"><span>'.__('Expired','webexpert-woocommerce-skroutz-smart-cart').'</span></div>';
			}elseif ($response->order->state=='cancelled') {
				echo '<div class="webexpert-skroutz-status cancelled"><span>'.__('Cancelled','webexpert-woocommerce-skroutz-smart-cart').'</span></div>';
			}elseif ($response->order->state=='returned') {
				echo '<div class="webexpert-skroutz-status returned"><span>'.__('Returned','webexpert-woocommerce-skroutz-smart-cart').'</span></div>';
			}elseif ($response->order->state=='partially_returned') {
				echo '<div class="webexpert-skroutz-status partially_returned"><span>'.__('Partially Returned','webexpert-woocommerce-skroutz-smart-cart').'</span></div>';
			}elseif ($response->order->state=='for_return') {
				echo '<div class="webexpert-skroutz-status for_return"><span>'.__('For return','webexpert-woocommerce-skroutz-smart-cart').'</span></div>';
			}

			echo "<div class='webexpert-field-label'>".sprintf(_x('Order: %s', 'the order key', 'webexpert-woocommerce-skroutz-smart-cart'), $order_key)."</div>";
			if ($response->order->courier_voucher) {
				echo "<div class='webexpert-field-label'>".__('Carrier','webexpert-woocommerce-skroutz-smart-cart').": ".$response->order->courier."</div>";
				echo "<div class='webexpert-field-label'>".__('Voucher','webexpert-woocommerce-skroutz-smart-cart').": <a target='_blank' href='".$response->order->courier_voucher."'>".reset($response->order->courier_tracking_codes)."</a></div>";
			}
			echo "<div style='clear:both'></div>";

			if ($response->order->state=='open') {
				echo '<div class="webexpert-field-label">';
				echo '<label for="smart_cart_order_pickup_location">'.__('Pickup location','webexpert-woocommerce-skroutz-smart-cart').'</label>';
				echo "<select id='smart_cart_order_pickup_location'>";
				foreach ($response->order->accept_options->pickup_location as $pickup_location) {
					echo "<option value='{$pickup_location->id}'>{$pickup_location->label}</option>";
				}
				echo "</select>";
				echo "</div>";

				echo '<div class="webexpert-field-label">';
				echo '<label for="smart_cart_order_pickup_window">'.__('Pickup window','webexpert-woocommerce-skroutz-smart-cart').'</label>';
				echo "<select id='smart_cart_order_pickup_window'>";
				foreach ($response->order->accept_options->pickup_window as $pickup_window) {
					echo "<option value='{$pickup_window->id}'>{$pickup_window->label}</option>";
				}
				echo "</select>";
				echo "</div>";
			}

			echo "<div class='webexpert-field-label'>";
			if ($response->order->state=='open') {
				echo "<button type='button' class='button accept-skroutz has-spinner' id='smart_cart_accept_order' data-order='" . $order_key . "'>" . __('Accept', 'webexpert-woocommerce-skroutz-smart-cart') . "</button>";
			}
			if ($response->order->state=='open' || $response->order->state=='accepted') {
				echo "<button type='button' class='button reject-skroutz' id='smart_cart_reject_toggle'>" . __('Reject', 'webexpert-woocommerce-skroutz-smart-cart') . "</button>";
			}
			echo "<div style='clear:both'></div>";

			echo "</div>";
			echo "<div style='clear:both'></div>";
			if ($response->order->state=='open' || $response->order->state=='accepted') {
				echo "<div id='smart_cart_order_reject_container'>";
				if ($response->order->state=='open') {
					echo "<div class='webexpert-field-label'>";
					echo "<select id='smart_cart_order_reject_reason'>";
					foreach ($response->order->reject_options->line_item_rejection_reasons as $line_item_rejection_reason) {
						echo "<option value='{$line_item_rejection_reason->id}'>{$line_item_rejection_reason->label}</option>";
					}
					echo "</select>";
					echo "</div>";

				}
				echo "<div class='webexpert-field-label'>";
				echo "<input type='text' placeholder='".__('Other reason','webexpert-woocommerce-skroutz-smart-cart')."' id='smart_cart_order_reject_reason_other'>";
				echo "<input type='hidden' id='smart_cart_order_line_numbers' value='".implode(',', array_column($response->order->line_items, 'id'))."'>";
				echo "</div>";
				if ($response->order->state=='open' || $response->order->state=='accepted') {
					echo "<button style='float:left' type='button' class='button reject-skroutz has-spinner' id='smart_cart_reject_order' data-order='" . $order_key . "'>" . __('Reject', 'webexpert-woocommerce-skroutz-smart-cart') . "</button>";
				}
				echo "<div style='clear:both'></div>";
				echo "</div>";
			}
			echo "</div>";
		}
	}

    function process_order( $response ) {
        $body = file_get_contents('php://input');
        $response = json_decode($body);

	    if (get_option('webexpert_skroutz_smart_cart_debug',null)=='yes') {
	        file_put_contents(wp_upload_dir()['basedir'].'/wc-logs/smart-cart.log',json_encode($response).PHP_EOL,FILE_APPEND);
        }

	    do_action('webexpert_skroutz_smart_cart_webhook_triggered',$response);
	    $this->create_new_order($response->order,!($response->event_type == "new_order" && $response->order->state == "open"));
    }

    function webexpert_skroutz_smart_cart_grab_order() {
	    $order_key=sanitize_text_field($_POST['order_id']);
	    $response=$this->fetch('orders/'.$order_key);
	    if ($response && isset($response->order)) {
		    if (get_option('webexpert_skroutz_smart_cart_debug',null)=='yes') {
			    file_put_contents(wp_upload_dir()['basedir'].'/wc-logs/smart-cart.log',json_encode($response).PHP_EOL,FILE_APPEND);
		    }

		    $order_id=$this->create_new_order($response->order,true);
		    wp_send_json(['success'=>1,'order_id'=>$order_id]);
	    }
	    wp_send_json(['success'=>0]);
    }

    function webexpert_skroutz_smart_cart_accept_order() {
	    $order_key=sanitize_text_field($_POST['order_id']);
	    $pickup_location=sanitize_text_field($_POST['pickup_location']);
	    $pickup_window=sanitize_text_field($_POST['pickup_window']);
	    $number_of_parcels=sanitize_text_field($_POST['number_of_parcels']);

	    $order_id=wc_get_order_id_by_order_key("SC-$order_key");
	    if ($order_id) {
	    	$order=wc_get_order($order_id);
		    $args=[
			    'number_of_parcels'=>$number_of_parcels,
			    'pickup_location'=>$pickup_location,
			    'pickup_window'=>$pickup_window,
		    ];

		    $response=$this->fetch('orders/'.$order_key.'/accept','POST',$args);

		    if (isset($response->errors) && sizeof($response->errors)>0) {
			    $errors=implode(',', array_map(function($x) { return implode(', ',$x->messages); }, $response->errors));
			    $note = sprintf(_x('Order %s has failed to be accepted on Skroutz smart cart. Errors: %s', 'the order key', 'webexpert-woocommerce-skroutz-smart-cart'), $order_key, $errors);
			    $order->add_order_note( $note );
			    $order->save();
			    wp_send_json(['success'=>0,'errors'=>$errors]);
		    }

		    if ($response->success) {
			    $note = sprintf(_x('Order %s has been approved at Skroutz smart cart.', 'the order key', 'webexpert-woocommerce-skroutz-smart-cart'), $order_key);
			    $order->add_order_note( $note );
			    $order->save();
			    $order->update_status('processing');
			    wp_send_json(['success'=>1,'errors'=>null]);
		    }
	    }
    }

    function webexpert_skroutz_smart_cart_reject_order() {

	    $order_key=sanitize_text_field($_POST['order_id']);
	    $reject_reason=sanitize_text_field($_POST['reject_reason']);
	    $reject_reason_other=sanitize_text_field($_POST['reject_reason_other']);
	    $line_numbers=sanitize_text_field($_POST['line_numbers']);

	    $order_id=wc_get_order_id_by_order_key("SC-$order_key");
	    if ($order_id) {
		    $order = wc_get_order($order_id);

		    if (!empty($reject_reason_other)) {
			    $args['rejection_reason_other'] = $reject_reason_other;
		    }else {
			    $line_items=explode(",",$line_numbers);
			    $args['line_items'] = [];

			    foreach ($line_items as $line_item) {
				    array_push($args['line_items'],['id'=>$line_item,'reason'=>$reject_reason]);
			    }
		    }

		    $response = $this->fetch('orders/' . $order_key . '/reject', 'POST', $args);
		    if ($response->success) {
			    $note = sprintf(_x('Order %s has been rejected at Skroutz smart cart with reason: '.$reject_reason_other, 'the order key', 'webexpert-woocommerce-skroutz-smart-cart'), $order_key);
			    $order->add_order_note( $note );
			    $order->save();
			    $order->update_status('cancelled');
			    wp_send_json(['success'=>1,'errors'=>null]);
		    }

		    if (isset($response->errors) && sizeof($response->errors)>0) {
			    $errors=implode(',', array_map(function($x) { return implode(', ',$x->messages); }, $response->errors));
			    $note = sprintf(_x('Order %s has failed to be rejected on Skroutz smart cart. Errors: %s', 'the order key', 'webexpert-woocommerce-skroutz-smart-cart'), $order_key, $errors);
			    $order->add_order_note( $note );
			    $order->save();
			    wp_send_json(['success'=>0,'errors'=>$errors]);
		    }
	    }
    }

	function on_hold_to_processing($order_id,$old_status,$new_status) {
		if (get_option('webexpert_skroutz_smart_cart_manage_orders', null)!="yes")
			return;

		$order=wc_get_order($order_id);
		if ($order) {
			if ($order->get_created_via()=='skroutz_smart_cart') {
				$order_key=str_replace("SC-","",$order->get_order_key());
				$response=$this->fetch('orders/'.$order_key);

				if ($old_status=="on-hold" && $new_status=="processing") {
					if ($response->order->state=='open') {
						if (get_option('webexpert_skroutz_smart_cart_auto_accept_orders_pickup_window','first_pickup_window')=="last_pickup_window") {
							$args=[
								'number_of_parcels'=>reset($response->order->accept_options->number_of_parcels),
								'pickup_location'=>end($response->order->accept_options->pickup_location)->id,
								'pickup_window'=>end($response->order->accept_options->pickup_window)->id,
							];
						}else {
							$args=[
								'number_of_parcels'=>reset($response->order->accept_options->number_of_parcels),
								'pickup_location'=>reset($response->order->accept_options->pickup_location)->id,
								'pickup_window'=>reset($response->order->accept_options->pickup_window)->id,
							];
						}

						$response=$this->fetch('orders/'.$order_key.'/accept','POST',$args);

						if (isset($response->errors) && sizeof($response->errors)>0) {
							$errors=implode(',', array_map(function($x) { return implode(', ',$x->messages); }, $response->errors));
							$note = sprintf(_x('Order %s has failed to be accepted on Skroutz smart cart. Errors: %s', 'the order key', 'webexpert-woocommerce-skroutz-smart-cart'), $order_key, $errors);
							$order->add_order_note( $note );
							$order->save();
							$order->update_status('on-hold');
						}

						if ($response->success) {
							$note = sprintf(_x('Order %s has been approved at Skroutz smart cart.', 'the order key', 'webexpert-woocommerce-skroutz-smart-cart'), $order_key);
							$order->add_order_note( $note );
							$order->save();
						}
					}
				}
			}
		}
	}

	function auto_accept_order($order_id) {
		if (get_option('webexpert_skroutz_smart_cart_auto_accept_orders',null)=='yes') {
			$order=wc_get_order($order_id);
			if ($order && $order->get_created_via()=='skroutz_smart_cart' && $order->get_status()=='on-hold') {
				$order_key=str_replace("SC-","",$order->get_order_key());
                $response=$this->fetch('orders/'.$order_key);
                if (count($response->order->line_items) == count( $order->get_items() )) {
	                $order->update_status('processing');
                }
			}
		}
	}

	function redirect_mails($args){
		$to = $args['to'];
		$domain_name = substr(strrchr($to, "@"), 1);
		if ($domain_name=="skroutz.gr") {
			$args['to']='';
		}
		return $args;
	}

	function webex_find_variation($product_id, $attributes) {
		$product_data_store = new WC_Product_Data_Store_CPT();
		$product = new WC_Product($product_id);

		return $product_data_store->find_matching_product_variation($product, $attributes);
	}
}